/**
 * YouList App - Main Application Logic
 */

// State
let playlist = [];
let currentIndex = -1;
let playQueue = [];
let wakeLock = null;
let draggedElement = null;
const MAX_LINKS = 5;

// Instagram
const INSTAGRAM_USERNAME = 'alementoria';

// Extrair ID do vídeo do YouTube
function extractVideoId(url) {
  try {
    const parsed = new URL(url);
    if (parsed.hostname.includes("youtube.com")) {
      const id = parsed.searchParams.get("v");
      logger.debug('ID extraído (youtube.com)', { url, id });
      return id;
    } else if (parsed.hostname === "youtu.be") {
      const id = parsed.pathname.substring(1);
      logger.debug('ID extraído (youtu.be)', { url, id });
      return id;
    }
  } catch (e) {
    // Se não for URL, tenta validar como ID direto
    if (/^[a-zA-Z0-9_-]{11}$/.test(url)) {
      logger.debug('ID validado diretamente', { id: url });
      return url;
    }
    logger.warn('Falha ao extrair ID', { url, error: e.message });
  }
  return null;
}

// Title Case inteligente
function titleCase(str) {
  if (!str) return "";
  
  const lowerWords = [
    "de", "da", "do", "das", "dos", "e", "em", "por", "para", "com",
    "a", "o", "as", "os", "no", "na", "nos", "nas", "ao", "à", "às",
    "pelo", "pela"
  ];
  
  return str.toLowerCase()
    .split(/(\s+|—|-)/g)
    .map((word, idx) => {
      if (word.trim() === "") return word;
      const plain = word.trim();
      if (lowerWords.includes(plain) && idx !== 0) return word.toLowerCase();
      return word.charAt(0).toUpperCase() + word.slice(1);
    })
    .join('');
}

// Buscar título via oEmbed
async function fetchVideoTitle(id) {
  try {
    logger.debug('Buscando título do vídeo', { id });
    
    const res = await fetch(
      `https://www.youtube.com/oembed?url=https://www.youtube.com/watch?v=${id}&format=json`
    );
    
    if (!res.ok) throw new Error(`HTTP ${res.status}`);
    
    const data = await res.json();
    let title = data.title || "Vídeo sem título";
    
    title = titleCase(title);
    title = [...title].slice(0, 40).join('');
    
    logger.success('Título obtido', { id, title });
    return title;
  } catch (err) {
    logger.error('Falha ao buscar título', { id, error: err.message });
    return "Vídeo sem título";
  }
}

// Renderizar playlist COM DRAG & DROP - SEM LIXEIRA
function renderPlaylist() {
  const container = document.getElementById("playlistContainer");
  container.innerHTML = "";
  
  playlist.forEach((item, index) => {
    const div = document.createElement("div");
    div.className = "playlist-item";
    div.draggable = true;
    div.dataset.index = index;
    
    // HTML LIMPO: drag, checkbox, título (SEM BOTÃO)
    div.innerHTML = `
      <span class="drag-handle">⋮⋮</span>
      <span title="${item.title}" data-index="${index}">${item.title}</span>
      <div class="checkbox-container">
        <input type="checkbox" data-index="${index}">
        <span class="play-number">${index + 1}</span>
      </div>
    `;
    
    container.appendChild(div);
  });
  
  updatePlaylistTitle();
  initDragAndDrop();
  initCheckboxes();
  logger.debug('Playlist renderizada', { count: playlist.length });
}

// Atualizar título da playlist
function updatePlaylistTitle() {
  const title = document.getElementById("playlistTitle");
  const credits = MAX_LINKS - playlist.length;
  title.textContent = `Playlist: ${credits} créditos de ${MAX_LINKS} (Free)`;
}

// Atualizar numeração sequencial
function updatePlayNumbers() {
  const items = document.querySelectorAll('.playlist-item');
  
  items.forEach((item, index) => {
    const checkbox = item.querySelector('input[type="checkbox"]');
    const numberSpan = item.querySelector('.play-number');
    
    // Atualizar número sequencial
    numberSpan.textContent = index + 1;
    
    // Mostrar/ocultar número baseado na seleção
    if (checkbox.checked) {
      item.classList.add('selected');
    } else {
      item.classList.remove('selected');
    }
  });
}

// Drag and Drop
function initDragAndDrop() {
  const container = document.getElementById('playlistContainer');
  const items = container.querySelectorAll('.playlist-item');
  
  items.forEach((item) => {
    // Drag start
    item.addEventListener('dragstart', (e) => {
      draggedElement = item;
      item.classList.add('dragging');
      e.dataTransfer.effectAllowed = 'move';
    });
    
    // Drag end
    item.addEventListener('dragend', (e) => {
      item.classList.remove('dragging');
      container.querySelectorAll('.playlist-item').forEach(i => {
        i.classList.remove('drag-over');
      });
      
      // Reordenar array da playlist
      reorderPlaylist();
      updatePlayNumbers();
    });
    
    // Drag over
    item.addEventListener('dragover', (e) => {
      e.preventDefault();
      e.dataTransfer.dropEffect = 'move';
      
      const afterElement = getDragAfterElement(container, e.clientY);
      if (afterElement == null) {
        container.appendChild(draggedElement);
      } else {
        container.insertBefore(draggedElement, afterElement);
      }
    });
    
    // Drag enter
    item.addEventListener('dragenter', (e) => {
      if (item !== draggedElement) {
        item.classList.add('drag-over');
      }
    });
    
    // Drag leave
    item.addEventListener('dragleave', (e) => {
      item.classList.remove('drag-over');
    });
    
    // Drop
    item.addEventListener('drop', (e) => {
      e.preventDefault();
      item.classList.remove('drag-over');
    });
  });
}

function getDragAfterElement(container, y) {
  const draggableElements = [...container.querySelectorAll('.playlist-item:not(.dragging)')];
  
  return draggableElements.reduce((closest, child) => {
    const box = child.getBoundingClientRect();
    const offset = y - box.top - box.height / 2;
    
    if (offset < 0 && offset > closest.offset) {
      return { offset: offset, element: child };
    } else {
      return closest;
    }
  }, { offset: Number.NEGATIVE_INFINITY }).element;
}

// Reordenar array da playlist baseado na ordem visual
function reorderPlaylist() {
  const items = document.querySelectorAll('.playlist-item');
  const newPlaylist = [];
  
  items.forEach((item) => {
    const title = item.querySelector('span[title]').textContent;
    const playlistItem = playlist.find(p => p.title === title);
    if (playlistItem) {
      newPlaylist.push(playlistItem);
    }
  });
  
  playlist = newPlaylist;
  savePlaylist();
  logger.debug('Playlist reordenada', { newOrder: playlist.map(p => p.title) });
}

// Checkboxes
function initCheckboxes() {
  const checkboxes = document.querySelectorAll('.playlist-item input[type="checkbox"]');
  checkboxes.forEach(checkbox => {
    checkbox.addEventListener('change', updatePlayNumbers);
  });
}

// Salvar/Carregar playlist
function savePlaylist() {
  try {
    localStorage.setItem("playlist", JSON.stringify(playlist));
    logger.debug('Playlist salva', { count: playlist.length });
  } catch (e) {
    logger.error('Falha ao salvar playlist', { error: e.message });
  }
}

function loadPlaylist() {
  try {
    const saved = localStorage.getItem("playlist");
    if (saved) {
      const parsed = JSON.parse(saved);
      playlist = parsed.map(item => ({
        id: item.id,
        title: item.title || "Vídeo sem título"
      }));
      renderPlaylist();
      logger.info('Playlist carregada', { count: playlist.length });
    }
  } catch (e) {
    logger.error('Falha ao carregar playlist', { error: e.message });
  }
}

// Adicionar à playlist
async function addToPlaylist() {
  const input = document.getElementById("urlInput");
  const url = input.value.trim();
  
  if (!url) {
    logger.warn('URL vazia');
    return;
  }
  
  const id = extractVideoId(url);
  
  if (!id) {
    logger.error('Link inválido', { url });
    return alert("Link inválido. Certifique-se de que é um link válido do YouTube.");
  }
  
  if (playlist.length >= MAX_LINKS) {
    logger.warn('Limite de vídeos atingido', { limit: MAX_LINKS });
    return alert(`Versão gratuita: Limite de ${MAX_LINKS} vídeos atingido. Remova um vídeo para adicionar outro.`);
  }
  
  // Verificar duplicatas
  if (playlist.some(item => item.id === id)) {
    logger.warn('Vídeo duplicado', { id });
    return alert("Este vídeo já está na playlist.");
  }
  
  const title = await fetchVideoTitle(id);
  playlist.push({ id, title });
  
  input.value = "";
  renderPlaylist();
  savePlaylist();
  updatePlaylistTitle();
  
  logger.trackVideoAdded(id, title);
}

// Deletar selecionados
function deleteSelected() {
  const checkboxes = document.querySelectorAll("#playlistContainer input[type='checkbox']");
  const toKeep = [];
  let deletedCount = 0;
  
  checkboxes.forEach((cb, i) => {
    if (!cb.checked) {
      toKeep.push(playlist[i]);
    } else {
      deletedCount++;
    }
  });
  
  playlist = toKeep;
  renderPlaylist();
  savePlaylist();
  updatePlaylistTitle();
  
  logger.trackPlaylistAction('delete_selected', deletedCount);
}

// Modais
function confirmClearAll() {
  document.getElementById("confirmModal").style.display = "flex";
  logger.debug('Modal de confirmação aberto');
}

function closeConfirm() {
  document.getElementById("confirmModal").style.display = "none";
  logger.debug('Modal de confirmação fechado');
}

function clearAll() {
  const count = playlist.length;
  playlist = [];
  renderPlaylist();
  savePlaylist();
  closeConfirm();
  updatePlaylistTitle();
  
  logger.trackPlaylistAction('clear_all', count);
}

// Extensões
function openExtensionsModal() {
  document.getElementById('extensionsModal').style.display = 'flex';
  logger.debug('Modal de extensões aberto');
}

function closeExtensionsModal() {
  document.getElementById('extensionsModal').style.display = 'none';
  logger.debug('Modal de extensões fechado');
}

function installExtensionUrl(url) {
  window.open(url, '_blank');
  logger.trackEvent('extension_clicked', { url });
}

// Copiar Pix
function copyPix() {
  const chavePix = "faledemim@gmail.com";
  navigator.clipboard.writeText(chavePix).then(() => {
    const msg = document.getElementById("msgPix");
    if (msg) {
      msg.style.display = "block";
      setTimeout(() => { msg.style.display = 'none'; }, 2500);
    }
    logger.trackEvent('pix_copied');
  }).catch(err => {
    logger.error('Falha ao copiar Pix', { error: err.message });
  });
}

// Instagram
function followInstagram() {
  const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
  const linkToOpen = isMobile 
    ? `instagram://user?username=${INSTAGRAM_USERNAME}` 
    : `https://www.instagram.com/${INSTAGRAM_USERNAME}/`;
  
  window.open(linkToOpen, '_blank');
  localStorage.setItem('youlist_followed_insta', 'true');
  
  const btn = document.getElementById('instagramFollowBtn');
  if (btn) btn.style.display = 'none';
  
  logger.trackEvent('instagram_follow');
}

// Modo Escuro (Wake Lock)
async function ativarModoEscuro() {
  document.getElementById("darkOverlay").style.display = "flex";
  logger.trackDarkMode(true);
  
  // Wake Lock - Manter tela acesa
  if ('wakeLock' in navigator) {
    try {
      wakeLock = await navigator.wakeLock.request('screen');
      logger.success('Wake Lock ativado');
    } catch (err) {
      logger.error('Erro ao ativar Wake Lock', { error: err.message });
    }
  } else {
    logger.warn('Wake Lock não suportado');
  }
}

function desativarModoEscuro() {
  document.getElementById("darkOverlay").style.display = "none";
  logger.trackDarkMode(false);
  
  // Liberar Wake Lock
  if (wakeLock !== null) {
    wakeLock.release().then(() => {
      wakeLock = null;
      logger.success('Wake Lock liberado');
    });
  }
}

// Share Target (receber links externos)
function handleShareTarget() {
  const params = new URLSearchParams(window.location.search);
  const sharedUrl = params.get('url') || params.get('text') || params.get('title');
  
  if (sharedUrl) {
    logger.info('Link compartilhado recebido', { url: sharedUrl });
    document.getElementById("urlInput").value = sharedUrl;
    setTimeout(() => addToPlaylist(), 500);
    window.history.replaceState({}, document.title, window.location.pathname);
  }
}

// Inicialização
document.addEventListener("DOMContentLoaded", () => {
  logger.info('YouList iniciando...', { version: '2.0' });
  
  // Dark Mode Button
  const wrapperDiv = document.getElementById("darkModeBtnWrapper");
  if (wrapperDiv) {
    const buttonSpan = document.createElement("span");
    buttonSpan.id = "darkModeBtn";
    buttonSpan.innerHTML = '<span style="color:#facc15;">🌙</span> <span>Modo Escuro</span>';
    buttonSpan.title = "Ativar modo escuro (não dorme)";
    buttonSpan.onclick = ativarModoEscuro;
    wrapperDiv.appendChild(buttonSpan);
  }

  // Event Listeners
  document.getElementById('addBtn').addEventListener('click', addToPlaylist);
  document.getElementById('playAllBtn').addEventListener('click', playAll);
  document.getElementById('playSelectedBtn').addEventListener('click', playSelected);
  document.getElementById('deleteSelectedBtn').addEventListener('click', deleteSelected);
  document.getElementById('confirmClearBtn').addEventListener('click', confirmClearAll);
  document.getElementById('clearYesBtn').addEventListener('click', clearAll);
  document.getElementById('clearNoBtn').addEventListener('click', closeConfirm);
  document.getElementById('openExtBtn').addEventListener('click', openExtensionsModal);
  document.getElementById('closeExtBtn').addEventListener('click', closeExtensionsModal);
  document.getElementById('instagramFollowBtn').addEventListener('click', followInstagram);
  document.getElementById('copiarPixBtn').addEventListener('click', copyPix);
  document.getElementById('darkOverlay').addEventListener('click', desativarModoEscuro);

  // Extensões
  document.querySelectorAll('#extensionsModal .modal-buttons button').forEach(btn => {
    btn.addEventListener('click', () => installExtensionUrl(btn.getAttribute('data-url')));
  });

  // Click na playlist
  document.getElementById('playlistContainer').addEventListener('click', (e) => {
    const span = e.target.closest('span[title]');
    if (span && span.dataset && span.dataset.index) {
      const idx = parseInt(span.dataset.index);
      if (!isNaN(idx)) playVideo(idx);
    }
  });

  // Enter no input
  document.getElementById('urlInput').addEventListener('keypress', (e) => {
    if (e.key === 'Enter') addToPlaylist();
  });

  // Verificar se já seguiu no Instagram
  if (localStorage.getItem('youlist_followed_insta') === 'true') {
    const btn = document.getElementById('instagramFollowBtn');
    if (btn) btn.style.display = 'none';
  }

  // Carregar playlist salva
  loadPlaylist();
  
  // Handle Share Target
  handleShareTarget();
  
  logger.success('YouList iniciado com sucesso');
});