/**
 * YouList Logger System
 * Sistema de registro centralizado para monitoramento e debugging
 */

class YouListLogger {
  constructor() {
    this.logs = [];
    this.maxLogs = 1000;
    this.enableConsole = true;
    this.enableStorage = true;
    this.sessionId = this.generateSessionId();
    
    this.init();
  }

  init() {
    this.info('Logger inicializado', { sessionId: this.sessionId });
    this.loadStoredLogs();
    this.setupErrorHandling();
  }

  generateSessionId() {
    return `session_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
  }

  log(level, message, data = {}) {
    const entry = {
      timestamp: new Date().toISOString(),
      level,
      message,
      data,
      sessionId: this.sessionId,
      url: window.location.href,
      userAgent: navigator.userAgent
    };

    this.logs.push(entry);
    
    if (this.logs.length > this.maxLogs) {
      this.logs.shift();
    }

    if (this.enableConsole) {
      const style = this.getConsoleStyle(level);
      console.log(`%c[${level}] ${message}`, style, data);
    }

    if (this.enableStorage) {
      this.saveToStorage();
    }

    return entry;
  }

  getConsoleStyle(level) {
    const styles = {
      INFO: 'color: #06b6d4; font-weight: bold',
      WARN: 'color: #f59e0b; font-weight: bold',
      ERROR: 'color: #ef4444; font-weight: bold',
      SUCCESS: 'color: #10b981; font-weight: bold',
      DEBUG: 'color: #8b5cf6; font-weight: bold'
    };
    return styles[level] || 'color: #94a3b8';
  }

  info(message, data) { return this.log('INFO', message, data); }
  warn(message, data) { return this.log('WARN', message, data); }
  error(message, data) { return this.log('ERROR', message, data); }
  success(message, data) { return this.log('SUCCESS', message, data); }
  debug(message, data) { return this.log('DEBUG', message, data); }

  // Eventos específicos da aplicação
  trackEvent(eventName, properties = {}) {
    return this.log('INFO', `Evento: ${eventName}`, properties);
  }

  trackVideoAdded(videoId, title) {
    return this.trackEvent('video_added', { videoId, title });
  }

  trackVideoPlayed(videoId, title, index) {
    return this.trackEvent('video_played', { videoId, title, index });
  }

  trackPlaylistAction(action, count) {
    return this.trackEvent('playlist_action', { action, videoCount: count });
  }

  trackError(errorType, errorMessage, stack) {
    return this.error(`Erro: ${errorType}`, { message: errorMessage, stack });
  }

  trackPWAInstall() {
    return this.trackEvent('pwa_installed');
  }

  trackDarkMode(enabled) {
    return this.trackEvent('dark_mode', { enabled });
  }

  // Persistência
  saveToStorage() {
    try {
      const recentLogs = this.logs.slice(-100);
      localStorage.setItem('youlist_logs', JSON.stringify(recentLogs));
    } catch (e) {
      console.warn('Falha ao salvar logs', e);
    }
  }

  loadStoredLogs() {
    try {
      const stored = localStorage.getItem('youlist_logs');
      if (stored) {
        const parsed = JSON.parse(stored);
        this.logs = Array.isArray(parsed) ? parsed : [];
      }
    } catch (e) {
      console.warn('Falha ao carregar logs', e);
    }
  }

  // Captura automática de erros
  setupErrorHandling() {
    window.addEventListener('error', (e) => {
      this.trackError('Uncaught Error', e.message, e.error?.stack);
    });

    window.addEventListener('unhandledrejection', (e) => {
      this.trackError('Unhandled Promise', e.reason, e.reason?.stack);
    });
  }

  // Exportar logs
  exportLogs() {
    return {
      sessionId: this.sessionId,
      totalLogs: this.logs.length,
      logs: this.logs,
      exportedAt: new Date().toISOString()
    };
  }

  downloadLogs() {
    const data = this.exportLogs();
    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `youlist-logs-${Date.now()}.json`;
    a.click();
    URL.revokeObjectURL(url);
    this.success('Logs exportados com sucesso');
  }

  // Limpar logs
  clearLogs() {
    this.logs = [];
    localStorage.removeItem('youlist_logs');
    this.info('Logs limpos');
  }

  // Estatísticas
  getStats() {
    const stats = {
      total: this.logs.length,
      byLevel: {},
      sessionDuration: null
    };

    this.logs.forEach(log => {
      stats.byLevel[log.level] = (stats.byLevel[log.level] || 0) + 1;
    });

    if (this.logs.length > 0) {
      const first = new Date(this.logs[0].timestamp);
      const last = new Date(this.logs[this.logs.length - 1].timestamp);
      stats.sessionDuration = Math.round((last - first) / 1000);
    }

    return stats;
  }

  // Filtrar logs
  filterLogs(criteria) {
    return this.logs.filter(log => {
      if (criteria.level && log.level !== criteria.level) return false;
      if (criteria.startTime && new Date(log.timestamp) < new Date(criteria.startTime)) return false;
      if (criteria.endTime && new Date(log.timestamp) > new Date(criteria.endTime)) return false;
      if (criteria.search && !log.message.toLowerCase().includes(criteria.search.toLowerCase())) return false;
      return true;
    });
  }
}

// Instância global
const logger = new YouListLogger();

// Expor globalmente
window.YouListLogger = logger;