/**
 * YouList Player Module
 * Gerencia o YouTube IFrame Player
 */

let player;
let apiReady = false;

// Carregar YouTube IFrame API
(function loadYouTubeAPI() {
  const tag = document.createElement('script');
  tag.src = "https://www.youtube.com/iframe_api";
  document.head.appendChild(tag);
  logger.info('YouTube API carregando...');
})();

// Callback quando API estiver pronta
function onYouTubeIframeAPIReady() {
  player = new YT.Player('player', {
    height: '400',
    width: '100%',
    videoId: '',
    events: {
      onReady: onPlayerReady,
      onStateChange: onPlayerStateChange,
      onError: onPlayerError
    }
  });
}

function onPlayerReady(event) {
  apiReady = true;
  logger.success('YouTube Player pronto');
}

function onPlayerStateChange(event) {
  const states = {
    '-1': 'UNSTARTED',
    '0': 'ENDED',
    '1': 'PLAYING',
    '2': 'PAUSED',
    '3': 'BUFFERING',
    '5': 'CUED'
  };
  
  const stateName = states[event.data] || 'UNKNOWN';
  logger.debug('Player state changed', { state: stateName });

  // Quando começar a tocar
  if (event.data === YT.PlayerState.PLAYING) {
    let currentTitle = "Reproduzindo...";
    
    if (playlist[currentIndex]) {
      currentTitle = playlist[currentIndex].title;
      logger.trackVideoPlayed(
        playlist[currentIndex].id,
        playlist[currentIndex].title,
        currentIndex
      );
    } else if (player.getVideoData && player.getVideoData().title) {
      currentTitle = player.getVideoData().title;
    }
    
    updateMediaSession(currentTitle);
  }

  // Quando vídeo terminar
  if (event.data === YT.PlayerState.ENDED) {
    logger.info('Vídeo finalizado', { index: currentIndex });
    nextVideo();
  }
}

function onPlayerError(event) {
  const errors = {
    2: 'Request inválido',
    5: 'Erro HTML5 Player',
    100: 'Vídeo não encontrado',
    101: 'Vídeo não permitido para embed',
    150: 'Vídeo não permitido para embed'
  };
  
  const errorMsg = errors[event.data] || 'Erro desconhecido';
  logger.trackError('YouTube Player', errorMsg, `Error code: ${event.data}`);
  
  alert(`Erro no player: ${errorMsg}`);
}

// Controles do Player
function playVideo(index) {
  if (!apiReady) {
    logger.warn('Player não está pronto');
    return alert("Player ainda carregando...");
  }
  
  if (!playlist[index]) {
    logger.error('Índice de vídeo inválido', { index });
    return;
  }
  
  currentIndex = index;
  playQueue = [index];
  
  logger.info('Reproduzindo vídeo', {
    index,
    id: playlist[index].id,
    title: playlist[index].title
  });
  
  player.loadVideoById(playlist[index].id);
}

function playAll() {
  if (!apiReady) {
    logger.warn('Player não está pronto');
    return alert("Player ainda carregando...");
  }
  
  if (playlist.length === 0) {
    logger.warn('Playlist vazia');
    return alert("Nenhum vídeo na playlist.");
  }
  
  playQueue = playlist.map((_, i) => i);
  currentIndex = 0;
  
  logger.trackPlaylistAction('play_all', playlist.length);
  player.loadVideoById(playlist[playQueue[currentIndex]].id);
}

function playSelected() {
  if (!apiReady) {
    logger.warn('Player não está pronto');
    return alert("Player ainda carregando...");
  }
  
  const checkboxes = document.querySelectorAll("#playlistContainer input[type='checkbox']");
  playQueue = [];
  
  checkboxes.forEach((cb, i) => {
    if (cb.checked) playQueue.push(i);
  });
  
  if (playQueue.length === 0) {
    logger.warn('Nenhum vídeo selecionado para reprodução');
    return alert("Nenhum vídeo selecionado.");
  }
  
  currentIndex = 0;
  logger.trackPlaylistAction('play_selected', playQueue.length);
  player.loadVideoById(playlist[playQueue[currentIndex]].id);
}

function nextVideo() {
  currentIndex++;
  
  if (currentIndex < playQueue.length) {
    const nextId = playlist[playQueue[currentIndex]].id;
    logger.info('Próximo vídeo', { index: currentIndex, id: nextId });
    player.loadVideoById(nextId);
  } else {
    logger.info('Fila de reprodução finalizada');
  }
}

// Media Session API (Controles na notificação)
function updateMediaSession(title) {
  if ('mediaSession' in navigator) {
    navigator.mediaSession.metadata = new MediaMetadata({
      title: title,
      artist: 'YouList Premium',
      album: 'Minha Playlist',
      artwork: [
        { 
          src: 'https://cdn-icons-png.flaticon.com/512/1384/1384060.png', 
          sizes: '512x512', 
          type: 'image/png' 
        }
      ]
    });

    navigator.mediaSession.setActionHandler('play', () => {
      if (player && player.playVideo) {
        player.playVideo();
        logger.debug('Media Session: Play');
      }
    });
    
    navigator.mediaSession.setActionHandler('pause', () => {
      if (player && player.pauseVideo) {
        player.pauseVideo();
        logger.debug('Media Session: Pause');
      }
    });
    
    navigator.mediaSession.setActionHandler('nexttrack', () => {
      nextVideo();
      logger.debug('Media Session: Next');
    });
    
    logger.debug('Media Session atualizada', { title });
  }
}

// Atalho de teclado (Espaço = Play/Pause)
document.addEventListener('keydown', (e) => {
  if (e.code === 'Space' && e.target.tagName !== 'INPUT') {
    e.preventDefault();
    
    if (player && apiReady && typeof player.getPlayerState === 'function') {
      const state = player.getPlayerState();
      
      if (state === YT.PlayerState.PLAYING) {
        player.pauseVideo();
        logger.debug('Pausado via teclado');
      } else {
        player.playVideo();
        logger.debug('Reproduzido via teclado');
      }
    }
  }
});